﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Security.Cryptography;
using System.Text;
using System.Threading.Tasks;
using TMPro;
using UnityEngine;
using UnityEngine.PlayerLoop;
using UnityEngine.Tilemaps;

//TimeTrialMission represents and manages a time trial mission - get to a destination before time runs out
public class TimeTrialMission : Mission
{
    //TimeTrialDifficultyData is a container struct for difficulty specific parameters
    public struct TimeTrialDifficultyData
    {
        public int Time { get; private set; }
        public int Money { get; private set; }

        public TimeTrialDifficultyData(int time, int money)
        {
            Time = time;
            Money = money;
        }
    }

    public Vector3Int Destination { get; private set; }
    public Dictionary<Constants.MissionDifficulties, TimeTrialDifficultyData> DifficultyData { get; private set; } = new Dictionary<Constants.MissionDifficulties, TimeTrialDifficultyData>();

    //UI Elements
    private float _TimeRemaining = 0;
    private GameObject _BaseUIObject;
    private MissionTimerController _MissionTimer;
    private TMP_Text _DestinationText;

    public TimeTrialMission(Vector3Int missionSpawnPos) : base(missionSpawnPos)
    {

    }

    public override bool Generate()
    {
        try
        {
            //Let's get the possible destinations
            List<Vector3Int> possibleDestinations = GameController.Instance.LSystem.Drawer.RoadPositions;
            possibleDestinations.Shuffle();

            //Pick a random destination
            Randomizer.Regenerate();
            int index = Randomizer.RNG.Next(0, possibleDestinations.Count);
            Destination = possibleDestinations[index];

            //And get the data for difficulties
            DifficultyData[Constants.MissionDifficulties.Easy] = GenerateDataForDifficulty(ConfigurationManager.Instance.Missions.TimeTrial.Easy, ConfigurationManager.Instance.Missions.Core.Time.Easy);
            DifficultyData[Constants.MissionDifficulties.Medium] = GenerateDataForDifficulty(ConfigurationManager.Instance.Missions.TimeTrial.Medium, ConfigurationManager.Instance.Missions.Core.Time.Medium);
            DifficultyData[Constants.MissionDifficulties.Hard] = GenerateDataForDifficulty(ConfigurationManager.Instance.Missions.TimeTrial.Hard, ConfigurationManager.Instance.Missions.Core.Time.Hard);
            IsGenerated = true;
            return true;
        }

        catch(Exception ex)
        {
            Debug.LogError("ERROR: Caught an exception when generating time trial mission, returning false. The exception is: " + ex);
            return false;
        }
    }

    public override void StartMission(Constants.MissionDifficulties difficulty)
    {
        //Store the difficulty, update the minimap
        SelectedDifficulty = difficulty;
        MinimapManager.Instance.AddIcon(Destination, Constants.TimeTrialDestinationIconName, ColoursManager.Instance.Colours["MissionDestinationIcon"].Colour, ConfigurationManager.Instance.Missions.TimeTrial.DestinationIconRenderSize, 0, Resources.Load<Sprite>("Minimap/Icons/minimapDestinationIcon"));
       
        //Setup the UI
        _TimeRemaining = DifficultyData[SelectedDifficulty].Time;
        _BaseUIObject = GameManager.Instance.MissionsUIGameObject.FindChild("TimeTrial");
        _MissionTimer = _BaseUIObject.FindChild("HeaderFlyout").FindChild("Timer").GetComponent<MissionTimerController>();
        _DestinationText = _BaseUIObject.FindChild("DestinationText").GetComponent<TMP_Text>();

        //Let's get the street the destination is on to display in the UI
        if (StreetsManager.Instance.Streets.ContainsKey(Destination))
        {
            _DestinationText.SetText("Get to <color=yellow>" + StreetsManager.Instance.Streets[Destination] + "</color> before time runs out!");
            _DestinationText.gameObject.SetActive(true);
        }

        else
        {
            _DestinationText.gameObject.SetActive(false);
        }

        //Add warning to timer from 10s downwards
        for (int i = 10000; i >= 0; i -= 1000)
        {
            _MissionTimer.AddTime(i);
        }

        //Start the timer and the mission
        _MissionTimer.StartTimer(DifficultyData[SelectedDifficulty].Time);
        UpdateTimeUI();
        _BaseUIObject.SetActive(true);
        _IsMissionStarted = true;
    }

    public override void EndMission(bool passed)
    {
        //Stop the timer, tidy up the minimap
        _MissionTimer.StopTimer();
        MinimapManager.Instance.RemoveIcon(Constants.TimeTrialDestinationIconName);

        //And reward the player if they passed
        if (passed)
        {
            GameManager.Instance.IncrementMoney(DifficultyData[SelectedDifficulty].Money);
        }

        //Mission is ended
        _BaseUIObject.SetActive(false);
        _IsMissionStarted = false;
    }

    public override string GetRewardText()
    {
        if (SelectedDifficulty != Constants.MissionDifficulties.Sentinel)
        {
            return "+$" + DifficultyData[SelectedDifficulty].Money;
        }

        return "";
    }

    public override void SpawnMinimapIcon(string iconID)
    {
        MinimapManager.Instance.AddIcon(MissionSpawnPosition, iconID, ColoursManager.Instance.Colours["TimeTrialMissionIcon"].Colour, ConfigurationManager.Instance.Minimap.MissionIconRenderSize, 0, Resources.Load<Sprite>("Minimap/Icons/minimapTimeTrialIcon"));
    }

    public override void Update()
    {
        if(_IsMissionStarted)
        {
            //Decrement the time left, update the UI
            _TimeRemaining -= (Time.deltaTime * 1000.0f);
            UpdateTimeUI();

            if(_TimeRemaining <= 0.0f)
            {
                //Out of time, mission failed
                MissionsManager.Instance.EndMission(false);
            }

            else
            {
                Vector3Int tilePos = GameManager.Instance.RoadsTilemap.WorldToCell(GameManager.Instance.PlayerCarGameObject.transform.position);

                if(tilePos == Destination)
                {
                    //The player is at the destination, mission passed!
                    MissionsManager.Instance.EndMission(true);
                }
            }
        }
    }

    private TimeTrialDifficultyData GenerateDataForDifficulty(TimeTrialDifficultyConfiguration missionConfig, MissionsTimeDifficultyConfiguration timeConfig)
    {
        //Generate a reward amount of money
        Randomizer.Regenerate();
        int generatedMoney = Convert.ToInt32(Math.Round(Randomizer.RNG.Next(missionConfig.MoneyLowerRange, missionConfig.MoneyUpperRange) / (float)ConfigurationManager.Instance.Missions.Core.MoneyRoundFactor) * ConfigurationManager.Instance.Missions.Core.MoneyRoundFactor);

        //Generate an amount of time
        int generatedTime = GenerateTimeForDifficulty(timeConfig, MissionSpawnPosition, Destination);
        generatedTime = Convert.ToInt32(Math.Ceiling(generatedTime * (1.0f / ConfigurationManager.Instance.Missions.Core.Time.RoundFactor)) * ConfigurationManager.Instance.Missions.Core.Time.RoundFactor);

        return new TimeTrialDifficultyData(generatedTime, generatedMoney);
    }

    private void UpdateTimeUI()
    {
        _MissionTimer.UpdateTimer(_TimeRemaining);
    }
}
